<?php
/**
 * Builds out customizer options
 *
 * @package       Customizer_Library
 * @author        Devin Price
 */

if ( ! function_exists( 'customizer_library_register' ) ) : /**
 * Configure settings and controls for the theme customizer
 *
 * @since  1.0.0.
 *
 * @param  object $wp_customize The global customizer object.
 *
 * @return void
 */ {
	function customizer_library_register( $wp_customize ) {

		$customizer_library = Customizer_Library::Instance();

		$options = $customizer_library->get_options();

		// Bail early if we don't have any options.
		if ( empty( $options ) ) {
			return;
		}

		// Add the sections
		if ( isset( $options['sections'] ) ) {
			customizer_library_add_sections( $options['sections'], $wp_customize );
		}

		// Add the sections
		if ( isset( $options['panels'] ) ) {
			customizer_library_add_panels( $options['panels'], $wp_customize );
		}

		// Sets the priority for each control added
		$loop = 0;

		// Loops through each of the options
		foreach ( $options as $option ) {

			// Set blank description if one isn't set
			if ( ! isset( $option['description'] ) ) {
				$option['description'] = '';
			}

			if ( isset( $option['type'] ) ) {

				$loop ++;

				// Apply a default sanitization if one isn't set
				if ( ! isset( $option['sanitize_callback'] ) ) {
					$option['sanitize_callback'] = customizer_library_get_sanitization( $option['type'] );
				}

				// Set blank active_callback if one isn't set
				if ( ! isset( $option['active_callback'] ) ) {
					$option['active_callback'] = '';
				}

				// Set default mime_type to image.
				if ( ! isset( $option['mime_type'] ) ) {
					$option['mime_type'] = 'image';
				}

				// Set default flex_width to true.
				if ( ! isset( $option['flex_width'] ) ) {
					$option['flex_width'] = true;
				}

				// Set default flex_height to true.
				if ( ! isset( $option['flex_height'] ) ) {
					$option['flex_height'] = true;
				}

				// Set blank width if one isn't set
				if ( ! isset( $option['width'] ) ) {
					$option['width'] = '';
				}

				// Set blank height if one isn't set
				if ( ! isset( $option['height'] ) ) {
					$option['height'] = '';
				}

				// Add the setting
				customizer_library_add_setting( $option, $wp_customize );

				// Priority for control
				if ( ! isset( $option['priority'] ) ) {
					$option['priority'] = $loop;
				}

				// Adds control based on control type
				switch ( $option['type'] ) {

					case 'text':
					case 'url':
					case 'number':
					case 'select':
					case 'radio':
					case 'checkbox':
					case 'range':
					case 'dropdown-pages':

						$wp_customize->add_control(
							$option['id'], $option
						);

						break;

					case 'color':

						$wp_customize->add_control(
							new WP_Customize_Color_Control(
								$wp_customize, $option['id'], $option
							)
						);

						break;

					case 'media':

						$wp_customize->add_control(
							new WP_Customize_Media_Control(
								$wp_customize,
								$option['id'], array(
									'label'             => $option['label'],
									'section'           => $option['section'],
									'sanitize_callback' => $option['sanitize_callback'],
									'priority'          => $option['priority'],
									'active_callback'   => $option['active_callback'],
									'description'       => $option['description'],
									'mime_type'         => $option['mime_type']
								)
							)
						);

						break;

					case 'crop':

						$wp_customize->add_control(
							new WP_Customize_Cropped_Image_Control(
								$wp_customize,
								$option['id'], array(
									'label'             => $option['label'],
									'section'           => $option['section'],
									'sanitize_callback' => $option['sanitize_callback'],
									'priority'          => $option['priority'],
									'active_callback'   => $option['active_callback'],
									'description'       => $option['description'],
									'flex_width'        => $option['flex_width'],
									'flex_height'       => $option['flex_height'],
									'width'             => $option['width'],
									'height'            => $option['height'],
								)
							)
						);

						break;

					case 'textarea':

						// Custom control required before WordPress 4.0
						if ( version_compare( $GLOBALS['wp_version'], '3.9.2', '<=' ) ) :

							$wp_customize->add_control(
								new Customizer_Library_Textarea(
									$wp_customize, $option['id'], $option
								)
							);

						else :

							$wp_customize->add_control( 'setting_id', array(
								$wp_customize->add_control(
									$option['id'], $option
								)
							) );

						endif;

						break;

					case 'date':

						$wp_customize->add_control(
							new Customizer_Library_Date(
								$wp_customize, $option['id'], $option
							)
						);

						break;

					case 'editor':

						$wp_customize->add_control(
							new Customizer_Library_Editor(
								$wp_customize, $option['id'], $option
							)
						);

						break;

					case 'radio-buttonset':

						$wp_customize->add_control(
							new Customizer_Library_Buttonset(
								$wp_customize, $option['id'], $option
							)
						);

						break;

					case 'select2':

						$wp_customize->add_control(
							new Customizer_Library_Select(
								$wp_customize, $option['id'], $option
							)
						);

						break;

					case 'select2-multiple':

						$wp_customize->add_control(
							new Customizer_Library_Select_Multiple(
								$wp_customize, $option['id'], $option
							)
						);

						break;

					case 'switch':

						$wp_customize->add_control(
							new Customizer_Library_Switch(
								$wp_customize, $option['id'], $option
							)
						);

						break;

					case 'group-title':

						$wp_customize->add_control(
							new Customizer_Library_Group_Title(
								$wp_customize, $option['id'], $option
							)
						);

						break;

				}
			}
		}
	}
}

endif;

add_action( 'customize_register', 'customizer_library_register', 100 );

/**
 * Add the customizer sections
 *
 * @since  1.2.0.
 *
 * @param  array $option
 *
 * @return void
 */
function customizer_library_add_sections( $sections, $wp_customize ) {

	foreach ( $sections as $section ) {

		if ( ! isset( $section['description'] ) ) {
			$section['description'] = FALSE;
		}

		$wp_customize->add_section( $section['id'], $section );
	}

}

/**
 * Add the customizer panels
 *
 * @since  1.2.0.
 *
 * @param  array $option
 *
 * @return void
 */
function customizer_library_add_panels( $panels, $wp_customize ) {

	foreach ( $panels as $panel ) {

		if ( ! isset( $panel['description'] ) ) {
			$panel['description'] = FALSE;
		}

		$wp_customize->add_panel( $panel['id'], $panel );
	}

}


/**
 * Add the setting and proper sanitization
 *
 * @since  1.2.0.
 *
 * @param  array $option
 *
 * @return void
 */
function customizer_library_add_setting( $option, $wp_customize ) {

	$settings_default = array(
		'default'              => NULL,
		'option_type'          => 'theme_mod',
		'capability'           => 'edit_theme_options',
		'theme_supports'       => NULL,
		'transport'            => NULL,
		'sanitize_callback'    => 'wp_kses_post',
		'sanitize_js_callback' => NULL
	);

	// Settings defaults
	$settings = array_merge( $settings_default, $option );

	// Arguments for $wp_customize->add_setting
	$wp_customize->add_setting( $option['id'], array(
			'default'              => $settings['default'],
			'type'                 => $settings['option_type'],
			'capability'           => $settings['capability'],
			'theme_supports'       => $settings['theme_supports'],
			'transport'            => $settings['transport'],
			'sanitize_callback'    => $settings['sanitize_callback'],
			'sanitize_js_callback' => $settings['sanitize_js_callback']
		)
	);

}

/**
 * Get default sanitization function for option type
 *
 * @since  1.2.0.
 *
 * @param  array $option
 *
 * @return void
 */
function customizer_library_get_sanitization( $type ) {

	if ( 'select' == $type || 'select2' == $type || 'select2-multiple' == $type || 'radio' == $type || 'radio-buttonset' == $type ) {
		return 'customizer_library_sanitize_choices';
	}

	if ( 'checkbox' == $type || 'switch' == $type ) {
		return 'customizer_library_sanitize_checkbox';
	}

	if ( 'color' == $type ) {
		return 'sanitize_hex_color';
	}

	if ( 'text' == $type || 'textarea' == $type || 'editor' == $type || 'date' == $type  ) {
		return 'customizer_library_sanitize_text';
	}

	if ( 'url' == $type ) {
		return 'esc_url';
	}

	if ( 'range' == $type ) {
		return 'customizer_library_sanitize_range';
	}

	if ( 'dropdown-pages' == $type || 'number' == $type || 'media' == $type || 'crop' == $type ) {
		return 'absint';
	}

	// If a custom option is being used, return false
	return FALSE;
}
